function Results=photoheat
%Results=photoheat
%The function makes a fit of the evolution of concentration of the substance A by irradiation. 
% (the substance can be transformed between two forms A <-> B by both photochemical and thermal reactions)
%and calculates the quantum yields of the photochemical reactions A->B and B->A. 
%The function determines the value and confidence intervals of 
% both quantum yields and the initial concentration of the substance A (at the beginning of irradiation). 
%
%The matrix "Results" contains the results of the fit: 
% 1. line: quantum yield (QY) of the photoreaction A->B, standard error of the QY, 
%           halfwidth of the confidence interval for the level of conficend 0.95 
% 2. line: QY of the photoreaction B->A, standard error of this QY, halfwidth of the 0.95 confidence interval 
%
%The input data are defined in the script "photoheat_input_data.m". 
%
%The function photoheath requires further functions resampling.m, effectiveepsilon.m, process_ftA.m, 
% process_sum_of_squares_ftA.m and process_sum_of_squares_ftB.m and the script photoheat_input_data.


%--------------------------------------------------------------
%Loading of data and initial data adjustment: 

close all
gr=0; %If gr=1, some additional figures are shown that enable to control a part of data adjustment. 
photoheat_input_data; %File with input data 
NA=6.023e23; %Avogadro constant 


A=load(data); %Loading of time, molar number of photons and molar concentrations
A(:,2)=A(:,2)-A(1,2); %Shift of photons number so that it starts at zero 
A(:,1)=A(:,1)-A(1,1); %Shift of time so that it starts at zero 

%Controls of the matrix with input data: 
[n,pompoc]=size(A); 
if pompoc~=4 
 error('The matrix with input data must contain 4 columns.'); 
end
[nic,pompor]=sort(A(:,1));
if sum(pompor==[1:n].')~=n
 error('Incorrect order or values of measurement times.'); 
end
[nic,pompor]=sort(A(:,2));
if sum(pompor==[1:n].')~=n
 error('Incorrect order or values of impacted photons.'); 
end

%Conversion of input data to standard units: 
A(:,2)=A(:,2)*NA; %Conversion of molar photon number to standard photon number
A(:,3)=A(:,3)*NA*1e3; %Conversion of molar concentration to standard concentration [m-3]
A(:,4)=A(:,4)*NA*1e3; %Conversion of molar concentration to standard concentration [m-3]
A(:,2)=A(:,2)*split_ratio; %Conversion of photon flow to the neighboring cuvette to the photon flow to the cuvette with the explored sample 


%Molar absorption coefficients: 
ep=load(mol_abs_koef); %Loading of molar absorption coefficients 
ep(:,[2 3])=ep(:,[2 3])*0.1/NA; %Conversion to standard units [m2]
ep(:,[2 3])=ep(:,[2 3])*log(10); %Conversion to cross sections [m2] 

if gr==1 
 K1=figure; plot(ep(:,1),ep(:,2),'b',ep(:,1),ep(:,3),'r'); hold on; 
end

%Ordering of the matrix with cross sections: 
[nic,por]=sort(ep(:,1));
for j=1:length(ep(:,1)),
 a(j,:)=ep(por(j),:); 
end
ep=a; 


V=Vml/1e6; %Sample volume conversion to m3
L=Lcm/100; %Cuvette length conversion to m 


spek=load(spektrum); %Loading of irradiation spectrum 

%Ordering of the matrix with irradiation spectrum: 
a=[]; 
[nic,por]=sort(spek(:,1));
for j=1:length(spek(:,1)),
 a(j,:)=spek(por(j),:); 
end
spek=a; 

%Background subtraction: 
if length(vlndel_pozadi)==2
 if vlndel_pozadi(1) > vlndel_pozadi(2) %Checks whether the borders are not exchanged
  error('The two values in the vector "vlndel_pozadi" are probably exchanged.'); 
 end
 [nic,pp1]=min(abs(spek(:,1)-vlndel_pozadi(1)));
 [nic,pp2]=min(abs(spek(:,1)-vlndel_pozadi(2)));
 spek(:,2)=spek(:,2)-mean(spek(pp1:pp2,2));
 figure; plot(spek(:,1),spek(:,2),'b.',spek(pp1:pp2,1),spek(pp1:pp2,2),'g'); grid on; 
title('Irradiation spectrum after background subtraction'); 
end
%if gr==1
 K2=figure; plot(spek(:,1),spek(:,2),'b'); hold on; 
%end

%Check of the "vlndel_r" values: 
if vlndel_r(1)<ep(1,1) | vlndel_r(1)<spek(1,1) | vlndel_r(2)>ep(end,1) | vlndel_r(2)>spek(end,1)
 error('The wavelength range "vlndel_r" is outside the measured spectral range.'); 
end
if vlndel_r(1) > vlndel_r(2) 
 error('The two values in the vector "vlndel_r" are probably exchanged.'); 
end


%Re-sampling: 
%First, the most dense data are selected: 
[nic,poz1]=min(abs(ep(:,1)-vlndel_r(1)));
[nic,poz2]=min(abs(ep(:,1)-vlndel_r(2)));
ep_orez=ep(poz1:poz2,:);
nep=poz2-poz1+1;

[nic,poz1]=min(abs(spek(:,1)-vlndel_r(1)));
[nic,poz2]=min(abs(spek(:,1)-vlndel_r(2)));
spek_orez=spek(poz1:poz2,:);
nspek=poz2-poz1+1; 

if gr==1
 figure(K1); plot(ep_orez(:,1),ep_orez(:,2),'b.',ep_orez(:,1),ep_orez(:,3),'r.'); 
end
%figure(K2); plot(spek_orez(:,1),spek_orez(:,2),'g.'); 

%Re-sampling on the wavelengths of the most dense data: 
if nep>=nspek 
 ep=ep_orez; 
 spek=resampling(spek,ep_orez(:,1)); 
else
 spek=spek_orez; 
 pom1=resampling(ep(:,[1 2]),spek_orez(:,1)); 
 pom2=resampling(ep(:,[1 3]),spek_orez(:,1)); 
 ep=[pom1 pom2(:,2)]; 
end
 
if gr==1
 figure(K1); plot(ep(:,1),ep(:,2),'co',ep(:,1),ep(:,3),'mo'); 
end
figure(K2); plot(spek(:,1),spek(:,2),'go'); title('Spectrum of the selected irradiation peak')


%---------------------------------------------------------------
%Calculation of effective cross sections: 
PA=[spek ep(:,2)];
disp('Effective cross section of absorption of the substance A  [m2]:'); 
epA=effectiveepsilon(PA,gr)
if gr==1
 title('Substance A'); 
end

PB=[spek ep(:,3)];
disp('Effective cross section of absorption of the substance B  [m2]:'); 
epB=effectiveepsilon(PB,gr)
if gr==1
 title('Substance B'); 
end


%---------------------------------------------------------------
%Graph with measured data: 
H1=figure; plot(A(:,2),A(:,3)/(1000*NA),'b*',A(:,2),A(:,4)/(1000*NA),'g*'); grid on; 
xlabel('Number of impacted photons'); ylabel('Molar concentration [mol/l]'); 
title('Measured concentration'); 
legend('A','B'); 

H2=figure; plot(A(:,1),A(:,3)/(1000*NA),'b*',A(:,1),A(:,4)/(1000*NA),'g*'); grid on; 
xlabel('Time [s]'); ylabel('Molar concentration [mol/l]'); 
title('Measured concentration - temporal dependence'); 
legend('A','B'); 


%--------------------------------------------------------------
%Input estimation of quantum yields: 
disp('%%%%%%%%%% Input estimation of quantum yields: %%%%%%%%%%%%%%%%'); 

dt=A(2:end,1)-A(1:end-1,1); %Time intervals between measurements 
dq=A(2:end,2)-A(1:end-1,2); %Numbers of photons impacted between measurements
dnA=A(2:end,3)-A(1:end-1,3); %Changes of A and B concentrations between measurements 

nAmean=(A(2:end,3)+A(1:end-1,3))/2; %Approximate mean values of concentration of the substance A in the periods between measurements 
nBmean=(A(2:end,4)+A(1:end-1,4))/2; %Approximate mean values of concentration of the substance B in the periods between measurements 
a=1-exp(-(epA*nAmean+epB*nBmean)*L); %Approximate numbers of photons absorbed in the periods between measurements
pA=epA*nAmean./(epA*nAmean+epB*nBmean); %Approximate percentages of photons absorbed by the substance A 
pB=epB*nBmean./(epA*nAmean+epB*nBmean); %Approximate percentages of photons absorbed by the substance B 

kkA=a.*pA/V; 
kkB=a.*pB/V; 

dnAf=dnA + KAb*nAmean.*dt - KBa*nBmean.*dt; %Approximate changes of A concentration caused only by photochemistry (subtraction of thermal reaction) 
y=dnAf./dq;

disp('Input estimation of quantum yields: '); %Calculated from changes of A concentration between individual measurements (the A and B concentrations were estimated as mean values of concentrations measured before and after irradiationza) by means of minimalization of sum of sqares
kvAb0=(sum(kkA.*kkB)*sum(kkB.*y) - sum(kkB.^2)*sum(kkA.*y)) / (sum(kkA.^2)*sum(kkB.^2) - sum(kkA.*kkB)^2) 
kvBa0=(sum(kkA.^2)*sum(kkB.*y) - sum(kkA.*kkB)*sum(kkA.*y)) / (sum(kkA.^2)*sum(kkB.^2) - sum(kkA.*kkB)^2) 

%Calculation of confidence intervals of input estimations: 
if gr==1 
 yfitted=-kkA*kvAb0 + kkB*kvBa0; 
 figure; plot3(kkA,kkB,y,'b*'); hold on; plot3(kkA,kkB,yfitted,'go'); grid on; 
 Sodh=sum((y-yfitted).^2); %Sum of squares of residuals
 rozptodh=Sodh/(length(y)-2); %unbiased sample variance, 2 is the number of fitted parameters
  
 Fodh=[kkA kkB]; 
 F2odh=Fodh.'*Fodh;
 Fvodh=inv(F2odh);

 u=1.95996; %Critical value of the N(0,1) distribution of the confidence interval with the level of significance 0.05 

 disp('Standard error of the input estimation of the quantum yield of the photoreaction A -> B and the halfwidth of the confidence interval for the level of confidence 0.95: '); 
 odch_kvAb0=sqrt(Fvodh(1,1)*rozptodh);%/pomerderivace;
 odch_kvAb0_0_05=sqrt(Fvodh(1,1)*rozptodh)*u;%/pomerderivace;
 [odch_kvAb0 odch_kvAb0_0_05]
 
 disp('Standard error of the input estimation of the quantum yield of the photoreaction B -> A and the halfwidth of the confidence interval for the level of confidence 0.95: '); 
 odch_kvBa0=sqrt(Fvodh(2,2)*rozptodh);%/pomerderivace;
 odch_kvBa0_0_05=sqrt(Fvodh(2,2)*rozptodh)*u;%/pomerderivace;
 [odch_kvBa0 odch_kvBa0_0_05]
end

pause(0.001); 



%---------------------------------------------------------------
%Calculation of quantum yields - fit of quantum yields and of the initial substance A concentration: 
disp('%%%%%%%%%% Fit 1 - fit of quantum yields and of the initial concentration (of the substance A): %%%%%%%%%%%%%%%%'); 

%The fit (quantum yields and initial A concentration): 
par=[kvAb0 kvBa0 A(1,3)]; %Initial estimates 
pomoptMFE=10000; %Default: 200*numberofvariables
pomoptMI=10000;
opt=optimset('MaxFunEvals',pomoptMFE,'MaxIter',pomoptMI);
par=fminsearch(@(par) process_sum_of_squares_ftA(par,A,epA,epB,KAb,KBa,L,V,presnost),par,opt); %This is the fit. 
disp('Quantum yield of the photochemical reaction A -> B: '); 
kvAb=par(1)
disp('Quantum yield of the photochemical reaction B -> A: '); 
kvBa=par(2)
disp('Initial concentraion of the substance A: '); 
NA0=par(3) 
disp('The mean total concentration of both substances A a B: '); 
mean(A(:,3)+A(:,4))

%Graph of the fit: 
NB0=mean(A(:,3)+A(:,4))-NA0; %Initial concentration of the substance B 
M=process_ftA(kvAb,kvBa,KAb,KBa,NA0,NB0,epA,epB,L,V,A(:,[1 2]),presnost,0); %Calculation of the evolution of A and B concentrations 
figure(H1); hold on; plot(M(:,2),M(:,3)/(1000*NA),'b',M(:,2),M(:,4)/(1000*NA),'g');
title('Concentration - irradiation dependence'); legend('A','B','fit A','fit B'); 

figure(H2); hold on; plot(M(:,1),M(:,3)/(1000*NA),'b',M(:,1),M(:,4)/(1000*NA),'g');
title('Concentration - temporal dependence'); legend('A','B','fit A','fit B'); 

H3=figure; plot(A(:,2),A(:,3)*V,'b*',A(:,2),A(:,4)*V,'g*',M(:,2),M(:,3)*V,'b',M(:,2),M(:,4)*V,'g'); grid on; 
xlabel('Number of impacted photons'); ylabel('Number of molecules'); 
title('Evolution of numbers of molecules'); 
legend('A','B','fit A','fit B'); 

pause(0.001); 


%---------------------------------------------------------
%Calculation of confidence intervals:

S=process_sum_of_squares_ftA(par,A,epA,epB,KAb,KBa,L,V,presnost); %Sum of squares of resituals
rozpt=S/(length(A(:,1))-3); %unbiased sample variance, 3 is the number of fitted parameters

%Calculation of derivatives:
por=[];
for j=1:length(A(:,1)), 
 [nic,pom]=min(abs(M(:,2)-A(j,2))); 
 if M(pom,2)-A(j,2)>0
  mder(j)=M(pom-1,3) + (M(pom,3)-M(pom-1,3))*(A(j,2)-M(pom-1,2))/(M(pom,2)-M(pom-1,2));
 elseif M(pom,2)-A(j,2)<0
  mder(j)=M(pom,3) + (M(pom+1,3)-M(pom,3))*(A(j,2)-M(pom,2))/(M(pom+1,2)-M(pom,2));
 else 
  mder(j)=M(pom,3);
 end
 %por(j)=pom;
end
mder=mder.'; 

pomerderivace=1e23; %Will be used for change of columns of the matrix "F" so that they had similar order of magnitude. 

M2=process_ftA(kvAb+deltader,kvBa,KAb,KBa,NA0,NB0,epA,epB,L,V,A(:,[1 2]),presnost,0);
m2der=[]; 
for j=1:length(A(:,1)), 
 [nic,pom]=min(abs(M2(:,2)-A(j,2)));
 if M2(pom,2)-A(j,2)>0
  m2der(j)=M2(pom-1,3) + (M2(pom,3)-M2(pom-1,3))*(A(j,2)-M2(pom-1,2))/(M2(pom,2)-M2(pom-1,2));
 elseif M2(pom,2)-A(j,2)<0
  m2der(j)=M2(pom,3) + (M2(pom+1,3)-M2(pom,3))*(A(j,2)-M2(pom,2))/(M2(pom+1,2)-M2(pom,2));
 else 
  m2der(j)=M2(pom,3);
 end
end
m2der=m2der.'; 
der1=(m2der-mder)/deltader; %Derivative of the fit in the measured points with respect to the quantum yield kvAb 
der1=der1/pomerderivace; 
   
M2=process_ftA(kvAb,kvBa+deltader,KAb,KBa,NA0,NB0,epA,epB,L,V,A(:,[1 2]),presnost,0);
m2der=[]; 
for j=1:length(A(:,1)), 
 [nic,pom]=min(abs(M2(:,2)-A(j,2)));
 if M2(pom,2)-A(j,2)>0
  m2der(j)=M2(pom-1,3) + (M2(pom,3)-M2(pom-1,3))*(A(j,2)-M2(pom-1,2))/(M2(pom,2)-M2(pom-1,2));
 elseif M2(pom,2)-A(j,2)<0
  m2der(j)=M2(pom,3) + (M2(pom+1,3)-M2(pom,3))*(A(j,2)-M2(pom,2))/(M2(pom+1,2)-M2(pom,2));
 else 
  m2der(j)=M2(pom,3);
 end
end
m2der=m2der.';
der2=(m2der-mder)/deltader; %Derivative of the fit in the measured points with respect to the quantum yield kvBa
der2=der2/pomerderivace; 


M2=process_ftA(kvAb,kvBa,KAb,KBa,NA0+deltader*NA0,NB0-NA0*deltader,epA,epB,L,V,A(:,[1 2]),presnost,0);
m2der=[];
for j=1:length(A(:,1)), 
 [nic,pom]=min(abs(M2(:,2)-A(j,2)));
 if M2(pom,2)-A(j,2)>0
  m2der(j)=M2(pom-1,3) + (M2(pom,3)-M2(pom-1,3))*(A(j,2)-M2(pom-1,2))/(M2(pom,2)-M2(pom-1,2));
 elseif M2(pom,2)-A(j,2)<0
  m2der(j)=M2(pom,3) + (M2(pom+1,3)-M2(pom,3))*(A(j,2)-M2(pom,2))/(M2(pom+1,2)-M2(pom,2));
 else 
  m2der(j)=M2(pom,3);
 end
end
m2der=m2der.'; 
der3=(m2der-mder)/(deltader*NA0); %Derivative of the fit in the measured points with respect to the initial concentration NA0 


%Matrices with derivatives: 
F=[der1 der2 der3];
F2=F.'*F;
Fv=inv(F2);

u=1.95996; %Critical value of the N(0,1) distribution of the confidence interval with the level of significance 0.05 (i.e. level of confidence 0.95) 

%Results: 
disp('Standard error of the A -> B quantum yield and the half-width of the confidence interval for the level of confidence 0.95: '); 
odch_kvAb=sqrt(Fv(1,1)*rozpt)/pomerderivace;
odch_kvAb_0_05=sqrt(Fv(1,1)*rozpt)*u/pomerderivace;
[odch_kvAb odch_kvAb_0_05]
disp('Standard error of the B -> A quantum yield and the half-width of the confidence interval for the level of confidence 0.95: '); 
odch_kvBa=sqrt(Fv(2,2)*rozpt)/pomerderivace;
odch_kvBa_0_05=sqrt(Fv(2,2)*rozpt)*u/pomerderivace;
[odch_kvBa odch_kvBa_0_05]
disp('Standard error of the initial concentration of the substance A and the half-width of the confidence interval for the level of confidence 0.95: ');
dch_NA0=sqrt(Fv(3,3)*rozpt);
dch_NA0_0_05=sqrt(Fv(3,3)*rozpt)*u;
[dch_NA0 dch_NA0_0_05]

Results=[kvAb odch_kvAb odch_kvAb_0_05; kvBa odch_kvBa odch_kvBa_0_05];





return; 
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%The following part of the function can be used when the total concentration of substances A and B should be added to the fitted parameters. However, such a fit can be error-prone due to the high correlation between fitted quantities: 

%Vlastni vypocet kvantovych vytezku - fitovani kvantovych vytezku, pocatecni koncentrace latky A a celkove koncentrace A+B: 
disp('%%%%%%%%%% Fit 2 - fit kvantovych vytezku, pocatecni koncentrace (A) a celkove koncentrace (A+B): %%%%%%%%%%%%%%%%'); 

%Fitovani kvantovych vytezku a pocatecni koncentrace latky A: 
par=[kvAb kvBa NA0 mean(A(:,3)+A(:,4))];
pomoptMFE=10000; %Default: 200*numberofvariables
pomoptMI=10000;
opt=optimset('MaxFunEvals',pomoptMFE,'MaxIter',pomoptMI);
par=fminsearch(@(par) process_sum_of_squares_ftB(par,A,epA,epB,KAb,KBa,L,V,presnost),par,opt); 
disp('QY of the photochemical reaction A -> B: '); 
kvAb=par(1)
disp('QY of the photochemical reaction B -> A: '); 
kvBa=par(2)
disp('Initial concentration of the compound A: '); 
NA0=par(3) 
disp('Total concentration of compounds A and B: '); 
Ncelk=par(4) 

%Vygrafeni fitu: 
NB0=Ncelk-NA0; %Pocatecni koncentrace cis-azobenzenu
M=process_ftA(kvAb,kvBa,KAb,KBa,NA0,NB0,epA,epB,L,V,A(:,[1 2]),presnost,0);
figure(H1); hold on; plot(M(:,2),M(:,3)/(1000*NA),'c',M(:,2),M(:,4)/(1000*NA),'y');
legend('A','B','fit1 A','fit1 B','fit2 A','fit2 B'); 

figure(H2); hold on; plot(M(:,1),M(:,3)/(1000*NA),'c',M(:,1),M(:,4)/(1000*NA),'y');
legend('A','B','fit1 A','fit1 B','fit2 A','fit2 B'); 

figure(H3); hold on; plot(M(:,2),M(:,3)*V,'c',M(:,2),M(:,4)*V,'y'); grid on; 
legend('A','B','fit1 A','fit1 B','fit2 A','fit2 B'); 

pause(0.001); 

%---------------------------------------------------------
%Vypocet intervalu spolehlivosti:

S=process_sum_of_squares_ftB(par,A,epA,epB,KAb,KBa,L,V,presnost); %Vypocet sumy ctvercu odchylek
rozpt=S/(length(A(:,1))-4); %rezidualni rozptyl, 4 je pocet fitovanych parametru 

%Pocitani derivaci:
por=[]; mder=[]; 
for j=1:length(A(:,1)), 
 [nic,pom]=min(abs(M(:,2)-A(j,2))); 
 if M(pom,2)-A(j,2)>0
  mder(j)=M(pom-1,3) + (M(pom,3)-M(pom-1,3))*(A(j,2)-M(pom-1,2))/(M(pom,2)-M(pom-1,2));
 elseif M(pom,2)-A(j,2)<0
  mder(j)=M(pom,3) + (M(pom+1,3)-M(pom,3))*(A(j,2)-M(pom,2))/(M(pom+1,2)-M(pom,2));
 else 
  mder(j)=M(pom,3);
 end
end
mder=mder.'; 


M2=process_ftA(kvAb+deltader,kvBa,KAb,KBa,NA0,NB0,epA,epB,L,V,A(:,[1 2]),presnost,0);
m2der=[]; 
for j=1:length(A(:,1)), 
 [nic,pom]=min(abs(M2(:,2)-A(j,2)));
 if M2(pom,2)-A(j,2)>0
  m2der(j)=M2(pom-1,3) + (M2(pom,3)-M2(pom-1,3))*(A(j,2)-M2(pom-1,2))/(M2(pom,2)-M2(pom-1,2));
 elseif M2(pom,2)-A(j,2)<0
  m2der(j)=M2(pom,3) + (M2(pom+1,3)-M2(pom,3))*(A(j,2)-M2(pom,2))/(M2(pom+1,2)-M2(pom,2));
 else 
  m2der(j)=M2(pom,3);
 end
end
m2der=m2der.'; 
der1=(m2der-mder)/deltader; %Derivace fitu podle kvAb v merenych bodech 
der1=der1/pomerderivace; 
   
M2=process_ftA(kvAb,kvBa+deltader,KAb,KBa,NA0,NB0,epA,epB,L,V,A(:,[1 2]),presnost,0);
m2der=[]; 
for j=1:length(A(:,1)), 
 [nic,pom]=min(abs(M2(:,2)-A(j,2)));
 if M2(pom,2)-A(j,2)>0
  m2der(j)=M2(pom-1,3) + (M2(pom,3)-M2(pom-1,3))*(A(j,2)-M2(pom-1,2))/(M2(pom,2)-M2(pom-1,2));
 elseif M2(pom,2)-A(j,2)<0
  m2der(j)=M2(pom,3) + (M2(pom+1,3)-M2(pom,3))*(A(j,2)-M2(pom,2))/(M2(pom+1,2)-M2(pom,2));
 else 
  m2der(j)=M2(pom,3);
 end
end
m2der=m2der.';
der2=(m2der-mder)/deltader;
der2=der2/pomerderivace; 
%{
figure; plot(M(:,2),M(:,3),'b',M(:,2),M(:,4),'r',M2(:,2),M2(:,3),'g--',M2(:,2),M2(:,4),'m--'); 
hold on; plot(M(:,2),M(:,3),'b.',M2(:,2),M2(:,3),'g.')
hold on; plot(A(:,2),m2der,'go',A(:,2),mder,'bo'); 
%}

M2=process_ftA(kvAb,kvBa,KAb,KBa,NA0+deltader*NA0,NB0-NA0*deltader,epA,epB,L,V,A(:,[1 2]),presnost,0);
m2der=[];
for j=1:length(A(:,1)), 
 [nic,pom]=min(abs(M2(:,2)-A(j,2)));
 if M2(pom,2)-A(j,2)>0
  m2der(j)=M2(pom-1,3) + (M2(pom,3)-M2(pom-1,3))*(A(j,2)-M2(pom-1,2))/(M2(pom,2)-M2(pom-1,2));
 elseif M2(pom,2)-A(j,2)<0
  m2der(j)=M2(pom,3) + (M2(pom+1,3)-M2(pom,3))*(A(j,2)-M2(pom,2))/(M2(pom+1,2)-M2(pom,2));
 else 
  m2der(j)=M2(pom,3);
 end
end
m2der=m2der.'; 
der3=(m2der-mder)/(deltader*NA0);

M2=process_ftA(kvAb,kvBa,KAb,KBa,NA0,NB0+NA0*deltader,epA,epB,L,V,A(:,[1 2]),presnost,0);
m2der=[];
for j=1:length(A(:,1)), 
 [nic,pom]=min(abs(M2(:,2)-A(j,2)));
 if M2(pom,2)-A(j,2)>0
  m2der(j)=M2(pom-1,3) + (M2(pom,3)-M2(pom-1,3))*(A(j,2)-M2(pom-1,2))/(M2(pom,2)-M2(pom-1,2));
 elseif M2(pom,2)-A(j,2)<0
  m2der(j)=M2(pom,3) + (M2(pom+1,3)-M2(pom,3))*(A(j,2)-M2(pom,2))/(M2(pom+1,2)-M2(pom,2));
 else 
  m2der(j)=M2(pom,3);
 end
end
m2der=m2der.'; 
der4=(m2der-mder)/(deltader*NA0);
%{
figure; plot(M(:,2),M(:,3),'b',M(:,2),M(:,4),'r',M2(:,2),M2(:,3),'g--',M2(:,2),M2(:,4),'m--'); 
hold on; plot(M(:,2),M(:,3),'b.',M2(:,2),M2(:,3),'g.')
hold on; plot(A(:,2),m2der,'go',A(:,2),mder,'bo'); 
%}


%Upravy matice derivaci: 
F=[der1 der2 der3 der4];
F2=F.'*F;
Fv=inv(F2);

u=1.95996; %Critical value of the N(0,1) distribution of the confidence interval with the level of significance 0.05

%Results: 
disp('Standard error of the QY A -> B, and a halfwidth of the confidence interval for P = 0.95: '); 
odch_kvAb=sqrt(Fv(1,1)*rozpt)/pomerderivace;
odch_kvAb_0_05=sqrt(Fv(1,1)*rozpt)*u/pomerderivace;
[odch_kvAb odch_kvAb_0_05]
disp('Standard error of the QY B -> A, and a halfwidth of the confidence interval for P = 0.95: '); 
odch_kvBa=sqrt(Fv(2,2)*rozpt)/pomerderivace;
odch_kvBa_0_05=sqrt(Fv(2,2)*rozpt)*u/pomerderivace;
[odch_kvBa odch_kvBa_0_05]
disp('Standard error of the initial concentration of compound A, and a halfwidth of the confidence interval for P = 0.95: ');
dch_NA0=sqrt(Fv(3,3)*rozpt);
dch_NA0_0_05=sqrt(Fv(3,3)*rozpt)*u;
[dch_NA0 dch_NA0_0_05]
disp('Standard error of the total concentration of compounds A and B, and a halfwidth of the confidence interval for P = 0.95: ');
dch_Ncelk=sqrt(Fv(4,4)*rozpt);
dch_Ncelk_0_05=sqrt(Fv(4,4)*rozpt)*u;
[dch_Ncelk dch_Ncelk_0_05]
